import os
from datetime import datetime

import pandas as pd
from meteostat import Daily, Point  # pylint: disable=import-error


def fetch_10year_weather_data(latitude, longitude, end_year, output_dir):
    """
    Fetch 10 years of daily weather data using Meteostat and save it as yearly parquet files.

    :param latitude: Latitude of the location
    :param longitude: Longitude of the location
    :param end_year: The last year of the data range (e.g., 2023)
    :param output_dir: Directory to save the output files
    """
    # Create output directory if it doesn't exist
    os.makedirs(output_dir, exist_ok=True)

    # Create Point for the specified location
    location = Point(latitude, longitude)

    # Calculate start and end dates
    end_date = datetime(end_year, 12, 31)
    start_date = datetime(end_year - 9, 1, 1)

    # Fetch daily data for the entire period
    data = Daily(location, start_date, end_date)
    df = data.fetch()

    # Check if data was successfully retrieved
    if df.empty:
        print("No data available for the specified period and location.")
        return

    # Select and rename relevant columns
    df = df[["tavg", "pres", "prcp"]]
    df.columns = ["Temperature", "Pressure", "Precipitation"]

    # Ensure the index is datetime
    df.index = pd.to_datetime(df.index)

    # Process and save data for each year
    for year in range(end_year - 9, end_year + 1):
        year_df = df[df.index.year == year]

        if not year_df.empty:
            output_file = os.path.join(output_dir, f"weather_data_{year}.parquet")
            year_df.to_parquet(output_file)
            print(f"Data for {year} saved to {output_file}")
            print(f"  Records: {len(year_df)}")
            print(f"  Date Range: {year_df.index.min()} to {year_df.index.max()}")
        else:
            print(f"No data available for {year}")

    print("\nAll yearly files have been generated.")
    print("\nData types:")
    print(df.dtypes)


# Example usage for Boston, MA
LATITUDE = 42.3601
LONGITUDE = -71.0589
END_YEAR = 2023  # This will fetch data from 2014 to 2023
OUTPUT_DIR = "weather_files/weather_data"

fetch_10year_weather_data(LATITUDE, LONGITUDE, END_YEAR, OUTPUT_DIR)
