import pandas as pd
from scipy.signal import find_peaks

def perform_threshold_autolabeling(
    dataset_id, record_id, data_coordinator, additional_parameters, trim_1=None, trim_2=None
):
    print(f"autolabeling {record_id}")
    
    # Debug output for weather app threshold function
    print(f"WEATHER APP THRESHOLD FUNCTION DEBUG:")
    print(f"  Record ID: {record_id}")
    print(f"  Received additional_parameters: {additional_parameters}")
    print(f"  Parameter keys: {list(additional_parameters.keys()) if additional_parameters else 'None'}")
    print(f"  trim_1: {trim_1}, trim_2: {trim_2}")
    
    if record_id is None:
        return []
    signals_to_autolabel = additional_parameters["threshold_signals"]
    threshold = additional_parameters["threshold_autolabel_threshold"]
    
    print(f"  Extracted signals_to_autolabel: {signals_to_autolabel} (type: {type(signals_to_autolabel)})")
    print(f"  Extracted threshold: {threshold} (type: {type(threshold)})")
    record_data = data_coordinator.fetch_data_async(
        data_coordinator.data_folder, dataset_id, record_id, signals_to_autolabel, {}, trim_1=trim_1, trim_2=trim_2
    )
    
    print(f"  Fetched data - signals count: {len(record_data['signals']) if record_data and 'signals' in record_data else 'No data'}")
    
    if len(record_data["signals"]) == 0:
        return []
    signal = record_data["signals"][0]
    signal_data = signal["data"]
    times = signal["times"]
    
    print(f"  Signal data length: {len(signal_data)}")
    print(f"  Signal data sample (first 5): {signal_data[:5] if len(signal_data) > 5 else signal_data}")
    print(f"  Times sample (first 5): {times[:5] if len(times) > 5 else times}")

    labels = []
    start = None
    finish = None

    for value_index, value in enumerate(signal_data):
        if value > threshold and start is None:
            start = times[value_index]
        if value < threshold and start is not None:
            finish = times[value_index]
        if start is not None and finish is not None:
            label = {}
            for possible_label in data_coordinator.all_labels:
                label[possible_label] = None
            label["Anomaly"] = True
            label["T1"] = pd.Timestamp(start) - pd.Timedelta(seconds=10)
            label["T2"] = pd.Timestamp(finish) + pd.Timedelta(seconds=10)
            
            print(f"  Created label: T1={label['T1']} (type: {type(label['T1'])}), T2={label['T2']} (type: {type(label['T2'])})")
            
            labels.append(label)
            start = None
            finish = None
    return labels


def perform_temperature_peaks_autolabeling(
    dataset_id, record_id, data_coordinator, additional_parameters, trim_1=None, trim_2=None
):
    print(f"autolabeling {record_id}")
    if record_id is None:
        return []
    record_data = data_coordinator.fetch_data_async(
        data_coordinator.data_folder, dataset_id, record_id, ["Temperature"], {}, trim_1=trim_1, trim_2=trim_2
    )
    if len(record_data["signals"]) == 0:
        return []
    signal = record_data["signals"][0]
    signal_data = signal["data"]
    times = signal["times"]
    times = pd.to_datetime(times)
    formatted_dates = [dt.strftime("%Y-%m-%d") for dt in times]

    labels = []
    peaks, _ = find_peaks(signal_data)
    for peak in peaks:
        label = {}
        peak_value = formatted_dates[peak]
        for possible_label in data_coordinator.all_labels:
            label[possible_label] = None
        label["Peak"] = True
        label["T1"] = formatted_dates[peak]
        label["T2"] = formatted_dates[peak]
        labels.append(label)
    return labels


autolabeling_dictionary = {
    "temperature_peaks": {
        "function": perform_temperature_peaks_autolabeling,
        "parameters": {},
        "display_name": "Temperature Peaks",
    },
    "threshold": {
        "function": perform_threshold_autolabeling,
        "parameters": {
            "autolabel_threshold": {
                "display_name": "Threshold",
                "default": 0,
                "min": None,
                "max": None,
            },
            "signals": {
                "display_name": "Signals",
                "default": ["Temperature"],
                "options": {
                    "Temperature": "Temperature",
                    "Pressure": "Pressure",
                    "Precipitation": "Precipitation",
                },
            },
        },
        "display_name": "Threshold",
    },
}